<?php
/**
 * server-processing-register.php (versão com tratativas profissionais)
 * - Mantém 100% do comportamento e mensagens originais
 * - Responde SEMPRE em JSON (success/error)
 * - Verifica sucesso real do insert
 * - Converte todos os erros silenciosos em respostas úteis
 * - Loga detalhes (com máscara) para depuração
 */

session_start();
require_once $_SERVER['DOCUMENT_ROOT'] . '/wp-load.php';

define('REG_DEBUG_ENABLED', true); // ou false

// ======================================================
// Helpers de resposta JSON e logging seguro
// ======================================================
header('Content-Type: application/json; charset=utf-8');

function json_success(array $payload = [], int $http = 200) {
  http_response_code($http);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

function json_error(string $msg, array $extra = [], int $http = 400) {
  http_response_code($http);
  echo json_encode(array_merge(['error' => $msg], $extra), JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

function reg_debug_log(array $data) {

  if (!defined('REG_DEBUG_ENABLED') || REG_DEBUG_ENABLED !== true) return;
  
  // Ajuste o caminho se preferir outro local
  $file = __DIR__ . '/server_register_debug.log';

  // Máscara segura
  if (isset($data['payload']['cpf'])) {
    $cpf = preg_replace('/\D+/', '', $data['payload']['cpf']);
    if (strlen($cpf) === 11) {
      $data['payload']['cpf'] = substr($cpf, 0, 3) . '*****' . substr($cpf, -3);
    } else {
      $data['payload']['cpf'] = '***';
    }
  }
  if (isset($data['payload']['password'])) $data['payload']['password'] = '***';
  if (isset($data['payload']['confirmar-password'])) $data['payload']['confirmar-password'] = '***';

  @file_put_contents(
    $file,
    json_encode([
      'ts'      => date('Y-m-d H:i:s'),
      'note'    => $data['note'] ?? null,
      'payload' => $data['payload'] ?? null,
      'debug'   => $data['debug'] ?? null
    ], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES) . PHP_EOL,
    FILE_APPEND | LOCK_EX
  );
}

function safe_mysqli_error($conn) {
  if ($conn instanceof mysqli) {
    return $conn->error;
  }
  return 'database error';
}

// ======================================================
// Entrada
// ======================================================
$data = $_POST ?? [];
reg_debug_log(['note' => 'incoming_payload', 'payload' => $data]);

// ======================================================
// Validações iniciais (mantidas como no original)
// ======================================================

// Validação do CPF
if (!validateCPF($data['cpf'] ?? '')) {
  json_error('CPF inválido. Por favor, digite um CPF válido.');
}

// Validação da senha
if (($data['password'] ?? '') !== ($data['confirmar-password'] ?? '')) {
  json_error('As senhas não conferem. Por favor, digite a mesma senha nos dois campos.');
}

// Validação de existência de CPF
$statusExistente = existCPF($data, $database_connection);

if ($statusExistente === 'publish') {
  json_error(
    'CPF já cadastrado.',
    ['code' => 'cpf_exists'],
    200
  );
}

if ($statusExistente === 'draft') {
  // Busca dados do cadastro desativado
  $cpfCheck = $data['cpf'] ?? '';
  $stmt = $database_connection->prepare("
    SELECT cliente_uuid, email 
    FROM customers 
    WHERE cpf = ? AND status = 'draft' 
    LIMIT 1
  ");
  if ($stmt) {
    $stmt->bind_param('s', $cpfCheck);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res ? $res->fetch_assoc() : null;
    $stmt->close();
  } else {
    json_error('Não foi possível verificar o cadastro existente. Tente novamente.');
  }

  // Sem registro ou sem e-mail: não deixa reativar via sistema
  if (!$row || empty(trim($row['email']))) {
    json_error(
      'Não foi possível reativar a conta por falta de e-mail. Contate o supermercado.',
      [
        'reactivate' => false,
        'code'       => 'no_email'
      ],
      200
    );
  }

  $email = trim($row['email']);

  // Gera código aleatório
  $codigo_verificacao = random_int(100000, 999999);

  // Salva na sessão (15 minutos)
  $_SESSION['reactivate_code']    = $codigo_verificacao;
  $_SESSION['reactivate_expires'] = time() + (15 * 60);
  $_SESSION['reactivate_cpf']     = $cpfCheck;
  $_SESSION['reactivate_userid']  = $row['cliente_uuid'];

  // Monta e-mail (reutilizando mesmo formato do reset)
  $subject = "Código de verificação para reativar cadastro";
  $body = "
    <p>Identificamos que seu CPF já possui um cadastro desativado.</p>
    <p>Para reativar sua conta, utilize o código abaixo:</p>
    <p><strong>{$codigo_verificacao}</strong></p>
    <p>O código é válido por 15 minutos.</p>
    <p>Se você não solicitou esta ação, ignore esta mensagem.</p>
  ";

  // Usa sua função global (mesma do fluxo de senha)
  if (function_exists('send_email_global')) {
    $sent = send_email_global($email, $subject, $body);
  } else {
    $headers = ['Content-Type: text/html; charset=UTF-8'];
    $sent = wp_mail($email, $subject, $body, $headers);
  }

  if (!$sent) {
    json_error(
      'Não foi possível enviar o código de verificação para reativar a conta. Tente novamente.',
      ['reactivate' => false],
      200
    );
  }

  // mascara de e-mail pra exibir no front
  if (strpos($email, '@') !== false) {
    list($userPart, $domainPart) = explode('@', $email, 2);
    $maskedUser = substr($userPart, 0, 2) . str_repeat('*', max(1, strlen($userPart) - 2));
    $email_masked = $maskedUser . '@' . $domainPart;
  } else {
    $email_masked = $email;
  }

  // Aqui NÃO cadastra de novo: informa que é reativação + dados pro front
  json_error(
    'CPF já cadastrado, mas está desativado. Enviamos um código de verificação para o seu e-mail.',
    [
      'reactivate'   => true,
      'email_masked' => $email_masked
    ],
    200
  );
}

// ======================================================
// Cadastro
// ======================================================
$ok = cadastrar_usuario($data, $database_connection);

// Verificação real de sucesso:
// - $ok deve ser um ID válido
// - e o registro deve existir no banco
if ($ok && is_string($ok)) {
  $userID = $ok;
} elseif ($ok && is_numeric($ok)) {
  $userID = (string)$ok;
} else {
  // Quando a função não retorna ID, ainda assim faremos uma checagem por CPF
  $userID = null;
}

// Double-check por CPF (garante que realmente inseriu)
$cpfCheck = $data['cpf'];
$chk = $database_connection->prepare("SELECT cliente_uuid, status FROM customers WHERE cpf = ? LIMIT 1");
if ($chk) {
  $chk->bind_param('s', $cpfCheck);
  $chk->execute();
  $r = $chk->get_result();
  $row = $r ? $r->fetch_assoc() : null;
  $chk->close();

  if ($row && !empty($row['cliente_uuid'])) {
    $userID = (string)$row['cliente_uuid'];
  }
}

if (!$userID) {
  // Não inseriu de fato — retornamos erro significativo
  reg_debug_log([
    'note' => 'insert_not_persisted',
    'payload' => $data,
    'debug' => ['db_error' => safe_mysqli_error($database_connection)]
  ]);
  json_error('Não foi possível concluir o cadastro no momento. Tente novamente em instantes.');
}

// Sucesso real
json_success(['success' => 'Cadastro realizado com sucesso!']);

// ======================================================
// Funções originais (mantidas) com tratativas internas
// ======================================================
function cadastrar_usuario($data, $database_connection) {

  // implementação para cadastrar usuário no banco de dados
  $register_simpliefied = get_field('cadastro_simplificado', 'options');

  $nome         = $data['nome'] ?? '';
  $email        = $data['email'] ?? '';
  $cpf          = $data['cpf'] ?? '';
  $nascimento   = $data['nascimento'] ?? '';
  $genero       = $data['genero'] ?? '';
  $celular      = $data['celular'] ?? '';
  $cep          = $data['cep'] ?? '';
  $estado       = $data['estado'] ?? '';
  $cidade       = $data['cidade'] ?? '';
  $bairro       = $data['bairro'] ?? '';
  $rua          = $data['rua'] ?? '';
  $numero       = (int) ($data['numero'] ?? 0);
  $password     = md5($data['password'] ?? '');
  $loja         = '';
  $lgpd         = $data['lgpd'] ?? 0;

  // Normaliza nascimento (como no original)
  if (!empty($nascimento)) {
    if (strpos($nascimento, '/') !== false) {
      // formato DD/MM/AAAA
      list($dia, $mes, $ano) = explode('/', $nascimento);
      $nascimento = date('Y-m-d', mktime(0, 0, 0, (int)$mes, (int)$dia, (int)$ano));
    } else {
      // qualquer outro formato já aceito pelo PHP
      $ts = @strtotime($nascimento);
      $nascimento = $ts ? date('Y-m-d', $ts) : '0000-00-00';
    }
  } else {
    $nascimento = '0000-00-00';
  }

  if (!empty($data['indicador'])) {
    $indicador = $data['indicador'];
  } else {
    $indicador = 0;
  }
  
  # SE O MODELO DE ASSINATURA ESTIVER ATIVO
  $dataAtual = date('d-m-Y');
  if (get_field('cobrar_assinatura', 'options') == true) {
    if (get_field('ativacao_padrao', 'options') == 'ativos') {
      $diasPadrao = get_field('dias_padrao_ativacao', 'options');
      $dataVencimento = date('d-m-Y', strtotime($dataAtual . ' + ' . $diasPadrao . ' days'));
      $statusPDV = "S";
    } elseif (get_field('ativacao_padrao', 'options') == 'inativos') {
      $dataVencimento = $dataAtual;
      $statusPDV = "N";
    } else {
      $dataVencimento = $dataAtual;
      $statusPDV = "S";
    }
  } else {
    $statusPDV = "S";
    $dataVencimento = $dataAtual;
  }

  // Geocoding (pode falhar; tratamos)
  $lat_long = get_lat_long($rua, (string)$numero, $bairro, $cidade, $estado, $cep);
  if (!$lat_long || strpos($lat_long, ',') === false) {
    reg_debug_log([
      'note' => 'geocoding_failed',
      'payload' => compact('rua','numero','bairro','cidade','estado','cep'),
      'debug' => ['lat_long' => $lat_long]
    ]);
    // Ainda seguimos (como o original) — a escolha da loja tratará fallback
  }

  list($lat_usuario, $lng_usuario) = explode(',', $lat_long . ','); // evita notice

  // Termos de loja
  $lojas = get_terms(array(
    'taxonomy'   => 'loja',
    'hide_empty' => false,
  ));

  $menor_distancia = PHP_FLOAT_MAX;
  $loja_mais_proxima = null;

  // Calcula distância (mesmo do original)
  if (!function_exists('calcular_distancia')) {
    function calcular_distancia($lat1, $lon1, $lat2, $lon2) {
        $raio_terra = 6371; // km
        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);
        $a = sin($dLat / 2) * sin($dLat / 2) +
            cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
            sin($dLon / 2) * sin($dLon / 2);
        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
        return $raio_terra * $c;
    }
  }

  if (is_array($lojas) && !empty($lojas)) {
    foreach ($lojas as $loja_tax) {
      $loja_coords = get_field('coordenadas', $loja_tax); // "lat,lng"
      if ($loja_coords) {
          list($lat_loja, $lng_loja) = explode(',', $loja_coords);

          if (is_numeric($lat_usuario) && is_numeric($lng_usuario) && is_numeric($lat_loja) && is_numeric($lng_loja)) {
              $distancia = calcular_distancia((float)$lat_usuario, (float)$lng_usuario, (float)$lat_loja, (float)$lng_loja);
              if ($distancia < $menor_distancia) {
                  $menor_distancia = $distancia;
                  $loja_mais_proxima = $loja_tax->term_id;
              }
          } else {
              // Fallback igual ao original
              $loja_mais_proxima = $loja_tax->term_id;
          }
      }
    }
  }

  if ($loja_mais_proxima) {
    $loja = $loja_mais_proxima;
  } else {
    // Mensagem original era texto puro; agora devolvemos JSON (mantendo sentido)
    json_error('Nenhuma loja próxima encontrada.');
  }

  // =========================
  // Cadastro simplificado: preencher endereço com o endereço da loja mais próxima
  // =========================
  if ($register_simpliefied && $loja) {

    // ACF da taxonomia normalmente aceita o objeto $term ou 'loja_{$term_id}'
    // Vamos buscar o term também para garantir compatibilidade
    $term_loja = get_term((int)$loja, 'loja');

    // Se você salvou os campos diretamente na taxonomia:
    $cep          = get_field('cep', $term_loja) ?: get_field('cep', 'loja_' . $loja) ?: $cep;
    $numero       = get_field('numero', $term_loja) ?: get_field('numero', 'loja_' . $loja) ?: $numero;
    $bairro       = get_field('bairro', $term_loja) ?: get_field('bairro', 'loja_' . $loja) ?: $bairro;
    $cidade       = get_field('cidade', $term_loja) ?: get_field('cidade', 'loja_' . $loja) ?: $cidade;
    $estado       = get_field('estado', $term_loja) ?: get_field('estado', 'loja_' . $loja) ?: $estado;
    $rua          = get_field('rua', $term_loja) ?: get_field('rua', 'loja_' . $loja) ?: $rua;
    $complemento  = get_field('complemento', $term_loja) ?: get_field('complemento', 'loja_' . $loja) ?: '';

    // Ajustes do simplificado
    $email      = '';
    $nascimento = '2000/01/01';
    $genero     = null;

    // Se quiser, você pode também setar o lat_long direto da loja (melhor que geocoding)
    $coordenadas = get_field('coordenadas', $term_loja) ?: get_field('coordenadas', 'loja_' . $loja);
    if ($coordenadas) {
      $lat_long = $coordenadas; // "lat,lng"
      list($lat_usuario, $lng_usuario) = explode(',', $lat_long . ',');
    }
  }


  $community = get_data('admin_config', 'community_required');
  $ativo = 1;
  $wpp_group_status = 0;

  if ($community) {
    $ativo = 0;
  } else {
    $ativo = 1;
  }

  // Busca vínculo de grupo pelo celular normalizado
  $telefone = preg_replace('/^\(?(\d{2})\)?\s*(\d{4,5})-?\s*(\d{4})$/', '55$1$2$3', $celular);
  $query = "SELECT * FROM wpp_groups_related WHERE customer_id = '$telefone' AND status = 2 LIMIT 1";
  $result = $database_connection->query($query);

  if ($result && $result->num_rows > 0) {
    $row = $result->fetch_assoc();
    $id_related_group = $row['id'];
    $wpp_group_status = 1;
    $ativo = 1;
  }

  // Inserção do cliente
  $userID = insert_customers(
    $database_connection,
    '',
    $nome,
    $cpf,
    $celular,
    $email,
    $nascimento,
    $genero,
    $cep,
    $estado,
    $cidade,
    $bairro,
    $rua,
    $numero,
    $password,
    '0,00',
    $indicador,
    $loja,
    '0',
    $dataVencimento,
    $lat_long ?: '',
    '0000-00-00 00:00:00',
    date('Y-m-d H:i:s'),
    'publish',
    '0',
    $ativo,
    $wpp_group_status
  );

  if (!$userID) {
    // Falha silenciosa no insert_customers
    reg_debug_log([
      'note' => 'insert_customers_failed',
      'payload' => compact('nome','cpf','celular','email','nascimento','genero','cep','estado','cidade','bairro','rua','numero','loja'),
      'debug' => ['db_error' => safe_mysqli_error($database_connection)]
    ]);
    json_error('Falha ao cadastrar o usuário.');
  }

  $_SESSION['usuario'] = array(
    'id' => $userID
  );

  if ($wpp_group_status == 1) {
    $_sql = "UPDATE wpp_groups_related SET customer_id = '$userID', status = 1 WHERE id = '$id_related_group'";
    $_query = mysqli_query($database_connection, $_sql);
    if (!$_query) {
      reg_debug_log([
        'note' => 'update_wpp_groups_related_failed',
        'payload' => ['id_related_group' => $id_related_group, 'userID' => $userID],
        'debug' => ['db_error' => safe_mysqli_error($database_connection)]
      ]);
      // Não impedimos o cadastro por falha neste update — seguimos em frente
    }
  }

  // Lógica PDV/API — mantém igual, mas sem die() de HTML
  if (!$community || $wpp_group_status == 1) {
    if (get_status_api(10) == 1) {

      $retorno = save_user_api($nome, $cpf, $rua, $numero, $bairro, $cidade, $estado, $cep, $celular, $email, $genero);

      if ($retorno == 1) {

        $_sql = "UPDATE customers SET pdv = '1' WHERE cliente_uuid = '$userID'";
        $_query = mysqli_query($database_connection, $_sql);
        if (!$_query) {
          reg_debug_log([
            'note' => 'update_customers_pdv_1_failed',
            'payload' => ['userID' => $userID],
            'debug' => ['db_error' => safe_mysqli_error($database_connection)]
          ]);
        }
        statusEntidade($userID, $statusPDV);

      } else {

        $_sql = "UPDATE customers SET pdv = '0' WHERE cliente_uuid = '$userID'";
        $_query = mysqli_query($database_connection, $_sql);
        if (!$_query) {
          reg_debug_log([
            'note' => 'update_customers_pdv_0_failed',
            'payload' => ['userID' => $userID],
            'debug' => ['db_error' => safe_mysqli_error($database_connection)]
          ]);
        }

      }

    } else {

      $_sql = "UPDATE customers SET pdv = '0' WHERE cliente_uuid = '$userID'";
      $_query = mysqli_query($database_connection, $_sql);
      if (!$_query) {
        reg_debug_log([
          'note' => 'update_customers_pdv_0_failed_status_api_off',
          'payload' => ['userID' => $userID],
          'debug' => ['db_error' => safe_mysqli_error($database_connection)]
        ]);
      }

    }
  } else {

    $_sql = "UPDATE customers SET pdv = '0' WHERE cliente_uuid = '$userID'";
    $_query = mysqli_query($database_connection, $_sql);
    if (!$_query) {
      reg_debug_log([
        'note' => 'update_customers_pdv_0_failed_community_required',
        'payload' => ['userID' => $userID],
        'debug' => ['db_error' => safe_mysqli_error($database_connection)]
      ]);
    }

  }

  // Retorna ID para verificação pelo chamador
  return $userID;
}

// ======== Funções originais (mantidas) ========

function validateCPF($cpf) {
  // implementação da validação do CPF
  $cpf = preg_replace('/[^0-9]/', '', $cpf);
  if (strlen($cpf) != 11) return false;
  if (preg_match('/(\d)\1{10}/', $cpf)) return false;

  for ($i = 0, $j = 10, $soma = 0; $i < 9; $i++, $j--) {
    $soma += $cpf[$i] * $j;
  }
  $resto = $soma % 11;
  $dv1 = ($resto < 2) ? 0 : 11 - $resto;

  for ($i = 0, $j = 11, $soma = 0; $i < 10; $i++, $j--) {
    $soma += $cpf[$i] * $j;
  }
  $resto = $soma % 11;
  $dv2 = ($resto < 2) ? 0 : 11 - $resto;

  if ($cpf[9] != $dv1 || $cpf[10] != $dv2) return false;

  return true;
}

function existCPF($data, $database_connection) {
  $cpf = $data['cpf'] ?? '';

  $query = "SELECT status FROM customers WHERE cpf = ?";
  $stmt = $database_connection->prepare($query);
  if ($stmt) {
    $stmt->bind_param('s', $cpf);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result && $result->num_rows > 0) {
      $row = $result->fetch_assoc();
      $stmt->close();
      return $row['status']; // 'publish' ou 'draft'
    }
    $stmt->close();
  }

  return false; // não existe
}